#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>

#define USE_SDL
//#define USE_X11

#ifdef USE_X11
#include <X11/Xlib.h>
#include "Xfct.h"
#endif USE_X11

#ifdef USE_SDL
#include <SDL/SDL.h>
Uint32 sdl_init_video_options;
SDL_Surface *screen;
SDL_Event event;
#endif USE_SDL


/* [STRUCTURE] cord3d                                                       *
 * ------------------                                                       *
 * x : absolute coordinates on X axis                                       *
 * y : absolute coordinates on X axis                                       *
 * z : absolute coordinates on X axis                                       */
typedef struct {
	double x;
	double y;
	double z;
} cord3d;

/* [STRUCTURE] cube3d                                                       *
 * ------------------                                                       *
 * center : absolute coordinates for the center of the cube                 *
 * cube   : coordonate of each point of the cube center at 0,0,0            *
 * size   : size of the cube                                                *
 * alphax : Angus on X axis                                                 *
 * alphay : Angus on Y axis                                                 *
 * alphaz : Angus on Z axis                                                 */
typedef struct {
	int size;
	double alphax;
	double alphay;
	double alphaz;
	cord3d *cube;
	cord3d center;
} cube3d;

//#define DEBUG
//#define ISO_3D

#define W 800
#define H 600

#define ANG 128
#define PI 3.14159265358979323846


cube3d cube;
int camera_zoom;

/* [FONCTION] putpixel(x,y,col)                                             *
 * ----------------------------                                             *
 *  Draw a pixel on the screen                                              *
 *  x : Coordonate on X                                                     *
 *  y : Coordonate on Y                                                     *
 *  col : Color of the point                                                */
void putpixel(int x, int y, int col)
{
	#ifdef USE_X11
	buffer[y*W + x] = col;
	#endif USE_X11
	#ifdef USE_SDL
	((Uint16*)screen->pixels)[(x+y*screen->w)]=col;
	#endif USE_SDL
}

/* [FONCTION] putpixel3d(x,y,z,col)                                         *
 * --------------------------------                                         *
 *  Draw a pixel on the screen using putpixel and convert 3D to 2D before   *
 *  displaying.                                                             *
 *  x : Coordonate on X                                                     *
 *  y : Coordonate on Y                                                     *
 *  z : Coordonate on Z                                                     *
 *  col : Color of the point                                                */
void putpixel3d(int x, int y, int z, int col)
{
	int sx, sy;
        #ifndef ISO_3D
	sx = W / 2 + (int)(camera_zoom * x / (camera_zoom + z));
	sy = H / 2 + (int)(camera_zoom * y / (camera_zoom + z));
        #endif
        /* Isomettry 3D */
        #ifdef  ISO_3D
	sx = W / 2 + x + z / 2;
	sy = H / 2 + y + z / 2;
        #endif
	putpixel(sx, sy, col);
}
/* [FONCTION] draw_line(x1,y1,x2,y2,col)                                   *
 * -------------------------------------                                   *
 * x1 : Coordonate of point 1 on X                                         *
 * x2 : Coordonate of point 2 on X                                         *
 * y1 : Coordonate of point 1 on Y                                         *
 * y2 : Coordonate of point 2 on Y                                         */
int draw_line(int x1, int y1, int x2, int y2, int col)
{
	int x, y, dx, dy, incx, incy;
	int c, i;
	dx = abs(x2-x1);
	dy = abs(y2-y1);
	if (y2 - y1 > 0) incy = 1;
	else incy = -1;
	if (x2 - x1 > 0) incx = 1;
	else incx = -1;
	x = x1;
	y = y1;
	if (dx > dy)
	{
		c = 2*dy - dx;
		for (i = 0; i < dx; i ++)
		{
			putpixel(x,y,col);
			x += incx;
			if (c < 0) c += 2*dy;
			else 
			{
				y += incy;
				c += 2*dy - 2*dx;
			}
		}
	}
	else
	{
	        c = 2*dx - dy;
		for (i = 0; i < dy; i++)
		{
			putpixel(x,y,col);
			y += incy;
			if (c < 0) c += 2*dx;
			else 
			{
				x += incx;
				c += 2*dx - 2*dy;
			}
		}
	}
	return 0;
}

/* [FONCTION] draw_line3d(c1,c2,col)                                        *
 * --------------------------------                                        *
 *  Draw a line betwen c1 and c2 on the screen using putpixel3d             * 
 *  c1 : Coordonate of point c1                                             *
 *  c2 : Coordonate of point c2                                             *
 *  col : Color of the line                                                 */
int draw_line3d(cord3d c1, cord3d c2, int col)
{
	int x1, x2, y1, y2; 
	#ifdef ISO_3D
	x1 = W / 2 + c1.x + c1.z / 2;
	x2 = W / 2 + c2.x + c2.z / 2;
	y1 = H / 2 + c1.y + c1.z / 2;
	y2 = H / 2 + c2.y + c2.z / 2;
	#endif ISO_3D
	#ifndef ISO_3D
	x1 = W / 2 + (int)(camera_zoom * c1.x / (camera_zoom + c1.z)); 
	x2 = W / 2 + (int)(camera_zoom * c2.x / (camera_zoom + c2.z));
	y1 = H / 2 + (int)(camera_zoom * c1.y / (camera_zoom + c1.z));
	y2 = H / 2 + (int)(camera_zoom * c2.y / (camera_zoom + c2.z));
	#endif ISO_3D
	draw_line(x1, y1, x2, y2, col);
	return 0;
}


/* [FONCTION] create_cube(varname, x, y, z, size)                           *
 * ---------------------------------------------                           *
 *  Create and initalizing a cube                                           * 
 *  varname : pointer to a cube3d                                           *
 *  x : Coordonate of the center of the cube on X                           *
 *  y : Coordonate of the center of the cube on Y                           *
 *  z : Coordonate of the center of the cube on Z                           *
 *  size : Size of the cube                                                 */
int create_cube(cube3d *varname, int x, int y, int z, int size)
{
	varname->alphax = 0;
	varname->alphay = 0;
	varname->alphaz = 0;
	varname->center.x = x;
	varname->center.y = y;
	varname->center.z = z;
	varname->cube = (cord3d *)malloc(sizeof(cord3d)*8);
	if (varname->cube == NULL) return -1;
	varname->cube[0].x = - size/2;
	varname->cube[0].y = - size/2;
	varname->cube[0].z = - size/2;
	varname->cube[1].x = + size/2;
	varname->cube[1].y = - size/2;      
	varname->cube[1].z = - size/2; 
	varname->cube[2].x = + size/2;
	varname->cube[2].y = + size/2;   
	varname->cube[2].z = - size/2;
	varname->cube[3].x = - size/2;
	varname->cube[3].y = + size/2;   
	varname->cube[3].z = - size/2;
	varname->cube[4].x = - size/2;
	varname->cube[4].y = - size/2;      
	varname->cube[4].z = + size/2;   
	varname->cube[5].x = + size/2;
	varname->cube[5].y = - size/2;   
	varname->cube[5].z = + size/2;
	varname->cube[6].x = - size/2;
	varname->cube[6].y = + size/2;   
	varname->cube[6].z = + size/2;
	varname->cube[7].x = + size/2;
	varname->cube[7].y = + size/2;   
	varname->cube[7].z = + size/2;
	return 0;
}
	
/* [FONCTION] draw_cube(varname, x, col)                                    *
 * ------------------------------------                                     *
 *  Draw a cube                                                             * 
 *  varname : pointer to a cube3d                                           *
 *  col : Color of the cube                                                 */
int draw_cube(cube3d varname, int col)
{
	int i;
	cord3d _cube[8];
        #ifdef DEBUG 
	printf("* draw_cube()...\n");
	#endif DEBUG
	for (i = 0; i < 8; i++)
	{
		#ifdef DEBUG
		printf("   [%d] %f %f %f\n", i, varname.cube[i].x, varname.cube[i].y, varname.cube[i].z);
		#endif DEBUG
		_cube[i].x = varname.cube[i].x + varname.center.x;
		_cube[i].y = varname.cube[i].y + varname.center.y;
		_cube[i].z = varname.cube[i].z + varname.center.z;
	}
        draw_line3d(_cube[1], _cube[0], col);
        draw_line3d(_cube[1], _cube[2], col);
        draw_line3d(_cube[0], _cube[4], col);
        draw_line3d(_cube[0], _cube[3], col);
        draw_line3d(_cube[3], _cube[2], col);
        draw_line3d(_cube[6], _cube[7], col);
        draw_line3d(_cube[4], _cube[6], col);
        draw_line3d(_cube[5], _cube[7], col);        
        draw_line3d(_cube[6], _cube[3], col); 
        draw_line3d(_cube[4], _cube[5], col);
        draw_line3d(_cube[1], _cube[5], col);   
        draw_line3d(_cube[2], _cube[7], col);
	#ifdef DEBUG
	printf("* draw_cube() : done !\n");
	#endif
	return 0;
}

void move_cube(cube3d *_cube,int depx, int depy, int depz)
{
	_cube->center.x += depx;
	_cube->center.y += depy;
	_cube->center.z += depz;
}

void rotate_point(cord3d *point, double angx, double angy, double angz)
{
	cord3d rotated;
	#ifdef DEBUG
	printf("** rotate_point()... \n");
	#endif DEBUG
	if (angz != 0)
	{
		#ifdef DEBUG
		printf("   Rotation on Z : %f\n", angz);
		#endif DEBUG
		rotated.x = cos(angz)*point[0].x - sin(angz)*point[0].y;
		rotated.y = sin(angz)*point[0].x + cos(angz)*point[0].y;
		point[0].x = rotated.x;
		point[0].y = rotated.y;
	}
	if (angx != 0)
	{
		#ifdef DEBUG
		printf("   Rotation on X : %f\n", angz); 
		#endif DEBUG
		rotated.y = cos(angx)*point[0].y - sin(angx)*point[0].z;
		rotated.z = sin(angx)*point[0].y + cos(angx)*point[0].z;
        	point[0].y = rotated.y;
        	point[0].z = rotated.z;
	}
	if (angy != 0)
	{
		#ifdef DEBUG
		printf("   Rotate on Y : %f\n", angy); 
		#endif
		rotated.x = cos(angy)*point[0].x - sin(angy)*point[0].z;
		rotated.z = sin(angy)*point[0].x + cos(angy)*point[0].z;
		point[0].x = rotated.x;
        	point[0].z = rotated.z;
	}
	#ifdef DEBUG
	printf("** rotate_point() : done!\n");
	#endif DEBUG
}

void rotate_cube(cube3d *_cube, double angx, double angy, double angz)
{
	int i;
	#ifdef DEBUG
	printf("* rotate_cube()... \n");
	#endif DEBUG
	for (i = 0; i < 8; i++)
	{
		#ifdef DEBUG
		printf("   [%d] %.5f %.5f %.5f\n", i, _cube->cube[i].x, _cube->cube[i].y, _cube->cube[i].z);
		#endif DEBUG
		rotate_point(&_cube->cube[i], angx, angy, angz);
		#ifdef DEBUG
		printf("   => %.5f %.5f %.5f\n", _cube->cube[i].x, _cube->cube[i].y, _cube->cube[i].z);
		#endif DEBUG
	}
	#ifdef DEBUG
	printf("* rotate_cube() : done!\n");
	#endif
}

int analyse_parms(int argc, char *argv[])
{
	int i;
	for (i=0; i < argc; i++)
	{
		if (strcmp("-h", argv[i]) == 0) 
		  {
			  printf("Usage :\n");
			  printf("-h : display this help\n");
			  printf("-f : fullscreen (only when using SDL)\n");
			  exit(0);
		  }
		#ifdef USE_SDL
		if (strcmp("-f", argv[i]) == 0) sdl_init_video_options |= SDL_FULLSCREEN;
		#endif USE_SDL
	}
	return 0;
}

int main(int argc, char *argv[])
{
	int i, frames, done;
	time_t start, end;
	double time_elap;
	#ifdef USE_SDL
	sdl_init_video_options = SDL_SWSURFACE;
	#endif USE_SDL
	analyse_parms(argc, argv);
        #ifdef USE_SDL
	printf("o Using libSDL\n");
	screen = SDL_SetVideoMode(W, H, 0, sdl_init_video_options); 
	printf("o Initialising : %dx%dx%d\n",screen->w, screen->h, screen->format->BitsPerPixel);
        SDL_ShowCursor(0);
        #endif USE_SDL
	#ifdef USE_X11
	printf("o Using libX11\n");
	init_x(0,0,W,H,0,"X Is CooL !");
	#endif USE_X11
	
	done = 0;
	frames = 0;	
	camera_zoom = 512;
	
	create_cube(&cube, 0, 0, 0, 200);	
    	start=time(NULL);
	i = 0;
	while (done == 0)
	{
		i++;
		#ifdef USE_SDL
		SDL_PollEvent(&event);
		switch (event.type) 
		{
		 case SDL_QUIT: 
			done = 1;
			break;
		 case SDL_KEYDOWN:
			if (event.key.keysym.sym == SDLK_ESCAPE) done = 1;
			break;
		}
		#endif USE_SDL
		draw_cube(cube, 0);
		if (i < ANG)
		{
			rotate_cube(&cube, 0, 0, PI/ANG/2);
			draw_cube(cube, 4015);
		}
		if ((ANG < i) && (i <= ANG*2))
		{
			rotate_cube(&cube, 0, PI/ANG/2, 0);
			draw_cube(cube, 1004);
		}
		if ((ANG*2 < i) && (i <= ANG*3))
		{
			rotate_cube(&cube, PI/ANG/2, 0, 0);
			draw_cube(cube, 3004);
			
		}
		if (i > ANG*3) i = 0;
		frames++;
		#ifdef USE_SDL
		SDL_UpdateRect(screen, 0, 0, screen->w, screen->h);
		#endif USE_SDL
                #ifdef USE_X11
		refresh_x(W,H);
		#endif USE_X11
		if (difftime(time(NULL),start) >= 20) done = 1;
	}
	end=time(NULL);
	time_elap = difftime(end, start); 
	printf("----------------------------\n");
	printf("Time elapsaed : %.2fs\n", time_elap);	
	printf("Results : %.2f fps with %d frames\n", (double)(frames)/time_elap, frames);	
	#ifdef USE_SDL
	SDL_Quit();
        #endif
	return 0;
}
	
